#!/usr/bin/env python3
# vol4_discrete_gauge_wilson_loop/run.py
import argparse
import copy
import pathlib
import sys
from itertools import product

# ---- Make core modules importable regardless of CWD ----
HERE = pathlib.Path(__file__).resolve().parent                      # .../vol4_discrete_gauge_wilson_loop
PROJECT_ROOT = HERE.parent                                          # .../vol4-fractal-pivot-hypersurface
for p in (PROJECT_ROOT, PROJECT_ROOT / "orig"):
    s = str(p)
    if s not in sys.path:
        sys.path.insert(0, s)

import yaml  # type: ignore

# Import from PROJECT_ROOT/orig/simulation/*
from simulation.discrete_gauge import run_discrete_gauge  # noqa: E402


def _ensure_list(x, fallback):
    if x is None:
        return list(fallback)
    if isinstance(x, (list, tuple)):
        return list(x)
    return [x]


def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--config", default="configs/default.yaml")
    ap.add_argument("--output-dir", default="data/results/vol4_discrete_gauge_wilson_loop")
    ap.add_argument("--gauges", default=None)  # e.g. U1,SU2,SU3
    ap.add_argument("--L", default=None)       # e.g. 6,8,12
    ap.add_argument("--b", default=None)       # e.g. 2.5
    ap.add_argument("--k", default=None)       # e.g. 0.1
    ap.add_argument("--n0", default=None)      # e.g. 0.0
    args = ap.parse_args()

    cfg_master = yaml.safe_load(open(args.config, "r"))

    # sweep sources (YAML → CLI overrides)
    b_vals  = _ensure_list(args.b.split(",") if args.b else None,  cfg_master.get("b_values",  [3.0]))
    k_vals  = _ensure_list(args.k.split(",") if args.k else None,  cfg_master.get("k_values",  [0.25]))
    n0_vals = _ensure_list(args.n0.split(",") if args.n0 else None, cfg_master.get("n0_values", [0.0]))
    L_vals  = _ensure_list([int(x) for x in args.L.split(",")] if args.L else None,
                           cfg_master.get("L_values", [6, 8, 12]))

    b_vals  = [float(x) for x in b_vals]
    k_vals  = [float(x) for x in k_vals]
    n0_vals = [float(x) for x in n0_vals]
    L_vals  = [int(x)   for x in L_vals]

    # gauges (CLI → YAML → default)
    if args.gauges:
        gauges = [g.strip() for g in args.gauges.split(",") if g.strip()]
    else:
        kpt = cfg_master.get("kernel_path_template")
        gauges = list(kpt.keys()) if isinstance(kpt, dict) else ["U1", "SU2", "SU3"]

    out_root = pathlib.Path(args.output_dir)
    out_root.mkdir(parents=True, exist_ok=True)

    for L in L_vals:
        for (b, k, n0) in product(b_vals, k_vals, n0_vals):
            cfg = copy.deepcopy(cfg_master)
            cfg.setdefault("crossover_analysis", {})["gauge_groups"] = gauges

            print(f"[RUN] L={L}  b={b}  k={k}  n0={n0}  gauges={gauges}")

            job_dir = out_root / f"L{L}" / f"b{b:.4g}_k{k:.4g}_n0{n0:.4g}"
            job_dir.mkdir(parents=True, exist_ok=True)

            run_discrete_gauge(
                b=b, k=k, n0=n0, L=L, cfg=cfg, output_dir=str(job_dir),
            )


if __name__ == "__main__":
    main()
